import os
import shutil
from pathlib import Path
from datetime import datetime, timezone
from PIL import Image, ImageDraw, ImageFont
from mutagen.id3 import ID3, COMM, TALB, TPE1, WXXX, APIC
from mutagen.mp3 import MP3
from mutagen.wave import WAVE
from io import BytesIO

# === CONFIGURATION ===
OUTPUT_DIR = "output"
CURRENT_DIR = Path.cwd()
ICON_FILENAME = "dogicon.png"

STAMP_TEXT = """anthroentertainment.com
Donate: donate.anthroentertainment.com
Public Domain Gift (CC0)
Archived: {}""".format(datetime.now(timezone.utc).isoformat())

SUPPORTED_IMAGE_EXTS = ('.jpg', '.jpeg', '.png')
SUPPORTED_AUDIO_EXTS = ('.mp3', '.wav')


# === OUTPUT SETUP ===
output_path = CURRENT_DIR / OUTPUT_DIR
if output_path.exists():
    shutil.rmtree(output_path)
output_path.mkdir(parents=True, exist_ok=True)


# === FONT HANDLING ===
def get_font_for_image(image_height, min_size=20, max_size=48):
    size = int(image_height * 0.04)
    size = max(min_size, min(size, max_size))
    try:
        return ImageFont.truetype("arial.ttf", size)
    except:
        return ImageFont.load_default()


# === IMAGE STAMPING ===
def stamp_image_with_canvas(src_path: Path, dst_path: Path):
    with Image.open(src_path) as img:
        img = img.convert("RGB")
        width, height = img.size

        font = get_font_for_image(height)
        lines = STAMP_TEXT.splitlines()
        line_height = font.getbbox("Ag")[3]
        padding = 12
        header_height = (line_height * len(lines)) + (padding * 2)

        new_img = Image.new("RGB", (width, height + header_height), "white")
        draw = ImageDraw.Draw(new_img)

        y = padding
        for line in lines:
            text_width = draw.textlength(line, font=font)
            draw.text(((width - text_width) / 2, y), line, fill="black", font=font)
            y += line_height

        new_img.paste(img, (0, header_height))
        new_img.save(dst_path, quality=95)


# === AUDIO TAGGING ===
def tag_audio_file(src_path: Path, dst_path: Path):
    ext = src_path.suffix.lower()
    shutil.copy2(src_path, dst_path)

    try:
        if ext == ".mp3":
            audio = MP3(dst_path, ID3=ID3)
            if audio.tags is None:
                audio.add_tags()

            audio["TPE1"] = TPE1(encoding=3, text="Anthro Entertainment")
            audio["TALB"] = TALB(encoding=3, text="Public Domain Gift (CC0)")
            audio["COMM"] = COMM(encoding=3, lang="eng", desc="License", text=STAMP_TEXT)
            audio["WXXX:Website"] = WXXX(encoding=3, desc="Website", url="https://anthroentertainment.com")
            audio["WXXX:Donate"] = WXXX(encoding=3, desc="Donate", url="https://donate.anthroentertainment.com")

            # === Embed resized album art ===
            icon_path = Path(ICON_FILENAME)
            if icon_path.exists():
                with Image.open(icon_path) as img:
                    img = img.convert("RGB")
                    img.thumbnail((300, 300))
                    buffer = BytesIO()
                    img.save(buffer, format="JPEG")
                    image_data = buffer.getvalue()
                    audio.tags.add(APIC(
                        encoding=3,
                        mime="image/jpeg",
                        type=3,
                        desc="Cover",
                        data=image_data
                    ))

            audio.save()

        elif ext == ".wav":
            audio = WAVE(dst_path)
            audio["artist"] = "Anthro Entertainment"
            audio["title"] = "Public Domain Gift (CC0)"
            audio["comment"] = STAMP_TEXT
            audio["icon"] = "dogicon.anthroentertainment.com"
            audio.save()

    except Exception as e:
        print(f"⚠️ Audio metadata failed for {src_path}: {e}")


# === PROCESS EACH FILE ===
def process_file(file_path: Path):
    if file_path.name == Path(__file__).name:
        return
    if "itch.io" in file_path.parts or OUTPUT_DIR in file_path.parts:
        return

    ext = file_path.suffix.lower()
    dst_path = output_path / file_path.name  # FLAT output

    if ext in SUPPORTED_IMAGE_EXTS:
        stamp_image_with_canvas(file_path, dst_path)
    elif ext in SUPPORTED_AUDIO_EXTS:
        tag_audio_file(file_path, dst_path)


# === WALKER FUNCTION ===
def walk_and_stamp():
    exclude_dirs = {OUTPUT_DIR, "itch.io"}
    for root, dirs, files in os.walk(CURRENT_DIR):
        dirs[:] = [d for d in dirs if d not in exclude_dirs]
        for name in files:
            process_file(Path(root) / name)


# === ENTRY POINT ===
if __name__ == "__main__":
    walk_and_stamp()
    print("✅ All images stamped and audio tagged into flat 'output/' folder")
